<?php

namespace App\Http\Controllers\Inventory;

use App\Http\Controllers\Controller;
use App\Models\InventoryItems;
use App\Models\StockMovements;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class InventoryItemController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        $items = InventoryItems::where('clinic_id', $user->clinic_id)
            ->orderBy('name')
            ->paginate(20);

        $totalItemsCount = $items->total();
        $lowStockCount   = InventoryItems::where('clinic_id', $user->clinic_id)
            ->whereColumn('current_quantity', '<', 'min_quantity')
            ->count();
        $totalStockValue = InventoryItems::where('clinic_id', $user->clinic_id)
            ->sum(\DB::raw('current_quantity * purchase_price'));

        return view('admin.inventory.index', compact(
            'items',
            'totalItemsCount',
            'lowStockCount',
            'totalStockValue'
        ));
    }

    public function create()
    {
        $item = new InventoryItems();
        return view('admin.inventory.form', compact('item'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();

        $data = $request->validate([
            'name'            => ['required', 'string', 'max:255'],
            'category'        => ['nullable', 'string', 'max:255'],
            'unit'            => ['nullable', 'string', 'max:50'],
            'sku'             => ['nullable', 'string', 'max:255'],
            'min_quantity'    => ['required', 'integer', 'min:0'],
            'current_quantity'=> ['required', 'integer', 'min:0'],
            'purchase_price'  => ['required', 'numeric', 'min:0'],
            'sale_price'      => ['nullable', 'numeric', 'min:0'],
        ], [
            'name.required'         => 'من فضلك أدخل اسم الصنف.',
            'min_quantity.required' => 'من فضلك أدخل الحد الأدنى للكمية.',
            'current_quantity.required' => 'من فضلك أدخل الكمية الحالية.',
            'purchase_price.required' => 'من فضلك أدخل سعر الشراء.',
        ]);

        $data['clinic_id'] = $user->clinic_id;

        InventoryItems::create($data);

        return redirect()
            ->route('inventory.items.index')
            ->with('success', 'تم إضافة الصنف إلى المخزون بنجاح.');
    }

    public function edit($id)
    {
        $user = Auth::user();

        $item = InventoryItems::where('clinic_id', $user->clinic_id)
            ->findOrFail($id);

        return view('admin.inventory.form', compact('item'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();

        $item = InventoryItems::where('clinic_id', $user->clinic_id)
            ->findOrFail($id);

        $data = $request->validate([
            'name'            => ['required', 'string', 'max:255'],
            'category'        => ['nullable', 'string', 'max:255'],
            'unit'            => ['nullable', 'string', 'max:50'],
            'sku'             => ['nullable', 'string', 'max:255'],
            'min_quantity'    => ['required', 'integer', 'min:0'],
            'current_quantity'=> ['required', 'integer', 'min:0'],
            'purchase_price'  => ['required', 'numeric', 'min:0'],
            'sale_price'      => ['nullable', 'numeric', 'min:0'],
        ]);

        $item->update($data);

        return redirect()
            ->route('inventory.items.index')
            ->with('success', 'تم تحديث بيانات الصنف بنجاح.');
    }

    public function destroy($id)
    {
        $user = Auth::user();

        $item = InventoryItems::where('clinic_id', $user->clinic_id)
            ->findOrFail($id);


        $item->delete();

        return redirect()
            ->route('inventory.items.index')
            ->with('success', 'تم حذف الصنف من المخزون (أرشفة) بنجاح.');
    }

    public function applyMovement(Request $request, $id)
    {
        $user = Auth::user();

        $item = InventoryItems::where('clinic_id', $user->clinic_id)
            ->findOrFail($id);

        $data = $request->validate([
            'type'     => 'required|in:in,out,adjustment',
            'quantity' => 'required|integer|min:1',
            'reason'   => 'nullable|string|max:255',
        ]);

        DB::transaction(function () use ($item, $data, $user) {

            // إنشاء حركة مخزون
            StockMovements::create([
                'inventory_item_id' => $item->id,
                'branch_id'         => $user->branch_id,
                'type'              => $data['type'],
                'quantity'          => $data['quantity'],
                'reason'            => $data['reason'],
                'reference_type'    => null,
                'reference_id'      => null,
            ]);

            if ($data['type'] === 'in') {
                $item->current_quantity += $data['quantity'];
            } elseif ($data['type'] === 'out') {
                $item->current_quantity -= $data['quantity'];
                if ($item->current_quantity < 0) {
                    $item->current_quantity = 0;
                }
            }

            $item->save();
        });

        return back()->with('success', 'تم تحديث المخزون بنجاح.');
    }

}
